// Copyright (c) Trend Micro Inc. 2013, All Rights Reserved

package com.trendmicro.ds.restapisamples;

import javax.ws.rs.core.Response.Status;

import org.jboss.resteasy.client.ClientResponse;
import org.jboss.resteasy.client.ClientResponseFailure;
import org.jboss.resteasy.client.ProxyFactory;
import org.jboss.resteasy.client.core.executors.ApacheHttpClient4Executor;
import org.jboss.resteasy.plugins.providers.RegisterBuiltin;
import org.jboss.resteasy.spi.ResteasyProviderFactory;

import com.trendmicro.ds.platform.rest.api.IAuthenticationAPI;
import com.trendmicro.ds.platform.rest.message.error.ErrorMessage;
import com.trendmicro.ds.platform.rest.object.DSCredentials;

/**
 * Authenticate: This sample application is designed demonstrate how to correctly authenticate
 * to the Deep Security Manager REST API. Successfully authenticating to the API
 * is a required precursor to most subsequent REST API calls.
 */
public class Authenticate extends SampleBase {

	/**
    * The main entry point.
    */
	public static void main(String[] args) {
		Authenticate client = new Authenticate();
		if(!client.setArguments(args)) {
			client.printUsage("Invalid arguments found.");
			return;
		}
		client.doAuthenticate();
	}

	private static final String USAGE = "Authenticate <DSM URL> <username> <password>\n" +
		"This sample application can be used to authenticate to a Deep Security\n" +
		"Manager REST API.\n";

	public void doAuthenticate() {

		// A variable to store the session ID.
		String sID = null;

		// RESTEasy client framework initialization.
		// This initialization only needs to be done once per VM
		RegisterBuiltin.register(ResteasyProviderFactory.getInstance());

		// An object that will execute HTTP requests
		ApacheHttpClient4Executor executor = new ApacheHttpClient4Executor();

		// Create the object that will communicate with the authentication API.
		IAuthenticationAPI authClient = ProxyFactory.create(IAuthenticationAPI.class, getDsmUrl(), executor);

		// Create the object to pass to the authentication call.
		DSCredentials credentials = new DSCredentials();
		credentials.setUserName(getUsername());
		credentials.setPassword(getPassword());
		try {
			this.log("Attempting to authenticate to Manager REST API...");
			sID = authClient.login(credentials);
	
			this.log("Authentication successful.");
			this.log("Authentication session ID string recieved: " + sID);
		} catch (ClientResponseFailure e) {
			// This is a special type of exception that means the server threw
			// an exception because there was a problem with the credentials.
			// It's important to handle these exceptions explicitly or the
			// connection to the server won't be released back in to the
			// underlying connection pool, meaning any subsequent API calls would fail.
			// See the RESTEasy Client Framework documentation for more details. 
			ClientResponse<?> clientResponse = e.getResponse();

			// Try to parse the error response in to an instance of the special
			// ErrorMessage class and display the result.
			Status status = clientResponse.getResponseStatus();
			log("Server returned error status code " + status.getStatusCode() + " (" + status + ")");
			ErrorMessage errorMessage = clientResponse.getEntity(ErrorMessage.class);
			log("Returned error message: " + errorMessage.getMessage());

		} catch (Exception e) {
			// Some other error happened, usually related to network communication problems.
			log("There was an error during authentication.", e);
			
		} finally {
			if (sID != null) {
				// Make sure to always log out.
				this.log("");
				this.log("Ending session...");
				authClient.endSession(sID);
				this.log("End session successful.");

				// make sure the session ID isn't accidentally re-used.
				sID = null;
			}
		}
		
		// Cleanup: force the HTTP Client to close any open sockets
		executor.close();

	}

	/**
	 * Return a string indicating how this program is to be used.
	 */
	@Override
	protected String getUsage() {
		return USAGE;
	}
}
