// Copyright (c) Trend Micro Inc. 2013, All Rights Reserved

package com.trendmicro.ds.restapisamples;

/**
 * SampleBase: This class is designed to keep basic info and common utility functions used by the Deep Security REST API client
 * sample applications.
 */
public abstract class SampleBase {

	private static final int	MINIMUM_ARG_COUNT	= 3;

	/** The program arguments. */
	private String[]			arguments;

	/** The URL of Deep Security Manager. */
	private String				dsmUrl;

	/** The username to authenticate with. */
	private String				username;

	/** The password to authenticate with. */
	private String				password;

	protected SampleBase() {
	}

	/**
	 * Set the program arguments to use when running the sample.
	 * Subclasses that need to process extra command line arguments should
	 * override this method, call super.setArguments, and if that succeeds
	 * then perform their own processing.
	 * 
	 * @return <code>true</code> if the arguments were valid and successfully set,
	 * <code>false</code> otherwise.
	 */
	public boolean setArguments(String[] arguments) {
		if (arguments.length < MINIMUM_ARG_COUNT) {
			return false;
		}
		this.arguments = arguments;
		
		dsmUrl = arguments[0];
		username = arguments[1];
		password = arguments[2];
		
		// some basic sanitization of the DSM URL
		if (dsmUrl.indexOf("//") < 0) {
			dsmUrl = "https://" + dsmUrl;
		}
		if (dsmUrl.endsWith("/rest/")) {
			// should not have the extra '/' on the end of the URL
			dsmUrl = dsmUrl.substring(0, dsmUrl.length() - 1);
		} else if (dsmUrl.endsWith("/")) {
			// add the "rest" to the URL so that it looks like https://host:port/rest
			dsmUrl = dsmUrl + "rest";
		} else if (!dsmUrl.endsWith("/rest")) {
			// add "/rest" to the URL so that it looks like https://host:port/rest
			dsmUrl = dsmUrl + "/rest";
		}
		
		return true;
		
	}

	/**
	 * Display the program usage
	 */
	public void printUsage() {
		printUsage("");
	}

	/**
	 * Display the program usage with an optional message string also displayed.
	 */
	public void printUsage(String message) {
		if (message != null & message.length() > 0) {
			log(message);
		}
		log("Usage:");
		log(getUsage());
	}

	/**
	 * Log a string message to the console.
	 */
	protected void log(String output) {
		System.out.println(output);
	}

	/**
	 * Log a string message and an exception stack trace to the console.
	 */
	protected void log(String output, Throwable t) {
		System.out.println(output);
		t.printStackTrace(System.out);
	}

	/**
	 * Subclasses must return a string indicating how this program is to be used.
	 */
	protected abstract String getUsage();

	/**
	 * Return the program arguments. Can be used by subclasses that need to parse extra arguments.
	 */
	protected String[] getArguments() {
		return arguments;
	}

	/**
	 * Return the Deep Security Manager URL.
	 */
	protected String getDsmUrl() {
		return dsmUrl;
	}

	/**
	 * Return the username to use to authenticate to the Deep Security Manager REST API.
	 */
	protected String getUsername() {
		return username;
	}

	/**
	 * Return the password to use to authenticate to the Deep Security Manager REST API.
	 */
	protected String getPassword() {
		return password;
	}

}
