// Copyright (c) Trend Micro Inc. 2014, All Rights Reserved

package com.trendmicro.ds.restapisamples;

import java.util.Arrays;
import java.util.Calendar;
import java.util.Comparator;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ws.rs.core.Response.Status;

import org.jboss.resteasy.client.ClientResponse;
import org.jboss.resteasy.client.ClientResponseFailure;
import org.jboss.resteasy.client.ProxyFactory;
import org.jboss.resteasy.client.core.executors.ApacheHttpClient4Executor;
import org.jboss.resteasy.plugins.providers.RegisterBuiltin;
import org.jboss.resteasy.spi.ResteasyProviderFactory;

import com.trendmicro.ds.platform.rest.api.IAuthenticationAPI;
import com.trendmicro.ds.platform.rest.api.IMonitoringAPI;
import com.trendmicro.ds.platform.rest.message.error.ErrorMessage;
import com.trendmicro.ds.platform.rest.object.DSCredentials;
import com.trendmicro.ds.platform.rest.object.monitoring.TenantHostProtectionElement;
import com.trendmicro.ds.platform.rest.object.monitoring.TenantHostProtectionListing;
import com.trendmicro.ds.platform.rest.object.monitoring.TenantModuleUsageElement;

/**
 * Billing: This sample application shows a an example of using computer
 * protection information in a sample billing application
 */
public class Billing extends SampleBase {

	/**
	 * The main entry point.
	 */
	public static void main(String[] args) {
		Billing sample = new Billing();
		if (!sample.setArguments(args)) {
			sample.printUsage("Invalid arguments found.");
			return;
		}
		sample.runSample();
	}

	private static final String USAGE = "Billing <DSM URL> <username> <password>\n" +
			"This sample application shows how to use computer protection information\n" +
			"obtained from Deep Security Manager's REST API in a sample billing application.\n";

	/** Mapping from known protection module codes to a more human-friendly string */ 
	private static final Map<String, String> moduleNameToFriendlyName;
	static {
		moduleNameToFriendlyName = new HashMap<String, String>();
		moduleNameToFriendlyName.put("WRS", "Web Reputation");
		moduleNameToFriendlyName.put("IM", "Integrity Monitoring");
		moduleNameToFriendlyName.put("LI", "Log Inspection");
		moduleNameToFriendlyName.put("DPI", "Intrusion Prevention");
		moduleNameToFriendlyName.put("FW", "Firewall");
		moduleNameToFriendlyName.put("AM", "Anti-Malware");
	}
	
	private static final double MILLISECONDS_PER_HOUR = 60.0 * 60.0 * 1000.0;

	/**
	 * Return a string indicating how this program is to be used.
	 */
	@Override
	protected String getUsage() {
		return USAGE;
	}

	/**
	 * Run the sample code
	 */
	public void runSample() {

		// A variable to store the session ID.
		String sID = null;

		// RESTEasy client framework initialization.
		// This initialization only needs to be done once per VM
		RegisterBuiltin.register(ResteasyProviderFactory.getInstance());

		// An object that will execute HTTP requests
		ApacheHttpClient4Executor executor = new ApacheHttpClient4Executor();

		// Create the object that will communicate with the authentication API.
		IAuthenticationAPI authClient = ProxyFactory.create(IAuthenticationAPI.class, getDsmUrl(), executor);
		IMonitoringAPI monitoringApi = ProxyFactory.create(IMonitoringAPI.class, getDsmUrl(), executor);

		// Create the object to pass to the authentication call.
		DSCredentials credentials = new DSCredentials();
		credentials.setUserName(getUsername());
		credentials.setPassword(getPassword());
		try {
			// Authenticate to the REST API
			sID = authClient.login(credentials);
	
			// Set up search date range to search for the previous day
			Calendar oneDayAgoStart = new GregorianCalendar();
			oneDayAgoStart.set(Calendar.MILLISECOND, 0);
			oneDayAgoStart.set(Calendar.SECOND, 0);
			oneDayAgoStart.set(Calendar.MINUTE, 0);
			oneDayAgoStart.set(Calendar.HOUR, 0);
			oneDayAgoStart.add(Calendar.DAY_OF_YEAR, -1);
			
			Calendar oneDayAgoEnd = (Calendar)oneDayAgoStart.clone();
			oneDayAgoEnd.add(Calendar.DAY_OF_YEAR, 1);
			
			// Get all the protection records for the last day for the tenant
			// with ID 0 - the primary tenant.
			TenantHostProtectionListing protectionListing = monitoringApi.listHostProtection(null, Integer.valueOf(0), oneDayAgoStart.getTime(), oneDayAgoEnd.getTime(), sID);

			// Calculate and display information about protection received
			calculateProtectionReceived(protectionListing.getProtectionElements(), oneDayAgoStart, oneDayAgoEnd);
			
		} catch (ClientResponseFailure e) {
			// This is a special type of exception that means the server threw
			// an exception because there was a problem with the credentials.
			// It's important to handle these exceptions explicitly or the
			// connection to the server won't be released back in to the
			// underlying connection pool, meaning any subsequent API calls would fail.
			// See the RESTEasy Client Framework documentation for more details. 
			ClientResponse<?> clientResponse = e.getResponse();

			// Try to parse the error response in to an instance of the special
			// ErrorMessage class and display the result.
			Status status = clientResponse.getResponseStatus();
			log("Server returned error status code " + status.getStatusCode() + " (" + status + ")");
			ErrorMessage errorMessage = clientResponse.getEntity(ErrorMessage.class);
			log("Returned error message: " + errorMessage.getMessage());

		} catch (Exception e) {
			// Some other error happened, usually related to network communication problems.
			log("There was an error during authentication.", e);
			
		} finally {
			if (sID != null) {
				// Make sure to always log out.
				authClient.endSession(sID);
				
				// make sure the session ID isn't accidentally re-used.
				sID = null;
			}
		}
		
		// Cleanup: force the HTTP Client to close any open sockets
		executor.close();

	}

	/**
	 * Calculate the amount of protection time received by protection module.
	 *
	 * @param protectionList
	 * 		the list of protection records retrieved from Deep Security Manager
	 * 
	 * @param periodStart
	 * 		the start of the time interval over which to calculate protection received
	 * 
	 * @param periodEnd
	 * 		the end of the time interval over which to calculate protection received
	 */
	private void calculateProtectionReceived(List<TenantHostProtectionElement> protectionList, Calendar periodStart, Calendar periodEnd) {
		long minTime = periodStart.getTimeInMillis();
		long maxTime = periodEnd.getTimeInMillis();

		// Total protection received. This is tracked separately as it's possible
		// the agent was active on a computer with no protection modules active
		long totalProtectionReceived = 0;
		// Total protection received broken down by module
		Map<String, Long> protectionReceivedByModule = new HashMap<String, Long>(); 
		
		for (TenantHostProtectionElement protectionRecord : protectionList) {
			// Protection start time is the most recent of the time in the protection
			// record and the start of the billing time interval 
			long protectionStart = Math.max(minTime, protectionRecord.getProtectionStartDate().getTime());

			// Protection stop time is either the value from the protection record 
			// (if there is one it means protection stopped during the time interval)
			// or the end of the billing time interval 
			long protectionStop;
			if (protectionRecord.getProtectionStopDate() == null) {
				protectionStop = maxTime;
			} else {
				protectionStop = protectionRecord.getProtectionStopDate().getTime();
			}

			// Increment total protection duration
			long protectionDuration = protectionStop - protectionStart;
			totalProtectionReceived += protectionDuration;
			
			// Increment protection duration for each protection module
			for (TenantModuleUsageElement moduleUsage : protectionRecord.getModuleUsageList()) {
				if (moduleUsage.isModuleEnabled()) {
					// Only need to do the calculation if the module was actually enabled
					Long currentModuleTotal = protectionReceivedByModule.get(moduleUsage.getModuleName());
					if (currentModuleTotal == null) {
						currentModuleTotal = protectionDuration;
					} else {
						currentModuleTotal = currentModuleTotal + protectionDuration; 
					}
					protectionReceivedByModule.put(moduleUsage.getModuleName(), currentModuleTotal);
				}
			}
			
		}

		// Display the calculated overall and per-module protection times in hours
		// A real billing system would feed the numbers computed here in to some cost calculation.
		System.out.format("Billing time period from %s to %s\n", periodStart.getTime().toString(), periodEnd.getTime().toString());
		System.out.format("Total protection duration: %,.2f hours\n", totalProtectionReceived / MILLISECONDS_PER_HOUR );
		for (Map.Entry<String, Long> mapEntry : protectionReceivedByModule.entrySet()) {
			// Try to map internal module name to a friendly string
			String friendlyName = moduleNameToFriendlyName.get(mapEntry.getKey());
			if (friendlyName == null) {
				friendlyName = mapEntry.getKey();
			}
			
			System.out.format("%,.2f hours of %s protection received\n", mapEntry.getValue().doubleValue() / MILLISECONDS_PER_HOUR, friendlyName);
		}
		
		// Some billing scenarios bill by peak number of protected computers in the billing period
		int peakProtectedComputers = computePeakProtectedComputers(protectionList);
		System.out.format("Maximum number of computers protected at the same time: %d", peakProtectedComputers);
		
		/*
		 This is a fairly simple demo to show what is possible. More advanced scenarios include:
		 1. Rounding up protection time for each computer to the nearest hour.

		    One thing to be careful of when rounding up is that changing the 
		    protection a computer receives causes the old protection record
		    to receive a protection end date and a new protection record
		    is created with:
		    	-start date equal to the end date in the old record
		    	-the new protection module information
		    	-no protection end date
		    This may or may not need to be accounted for depending on the
		    rounding requirements.
		    
		 2. Breaking down protection based on cloud instance type.

		 	Deep Security Manager must be configured with a cloud connector
		 	for this information to be available in TenantModuleUsageElement.
		 	
		 */
	}

	/**
	 * Compute the maximum number of computers that were protected simultaneously.
	 * This requires computing the maximum number of overlapping protection time intervals.
	 * The idea is as follows: <br/>
	 * - create a flat list of every protection start or stop event <br/>
	 * - sort the list from earliest to latest <br/>
	 * - traverse the list, incrementing the protected computer count for
	 * 		each protection start event and decrementing it for each
	 * 		protection stop event <br/>
	 * - the desired value is the maximum protected computer count
	 */
	private int computePeakProtectedComputers(List<TenantHostProtectionElement> protectionList) {
		long now = System.currentTimeMillis();
		
		// First, create a flat list of everything that happened (start or stop) during the time period.
		ProtectionEvent[] protectionEvents = new ProtectionEvent[protectionList.size() * 2];
		for (int i = 0; i < protectionList.size(); ++i) {
			TenantHostProtectionElement protectionRecord = protectionList.get(i);
			protectionEvents[2*i] = new ProtectionEvent(protectionRecord.getProtectionStartDate().getTime(), true);
			if (protectionRecord.getProtectionStopDate() == null) {
				protectionEvents[2*i + 1] = new ProtectionEvent(now, false);
			} else {
				protectionEvents[2*i + 1] = new ProtectionEvent(protectionRecord.getProtectionStopDate().getTime(), false);
			}
		}

		// Now sort the list so we have the order in which the actions happened
		Arrays.sort(protectionEvents, new Comparator<ProtectionEvent>() {
			/* Comparator to sort ProtectionEvents from earliest to latest. Sorting is defined so that if
			 * one interval ends at the same time another begins, the end of one is ordered before the start
			 * of the other. */
			@Override
			public int compare(ProtectionEvent o1, ProtectionEvent o2) {
				long timeDiff = o1.time - o2.time;
				if (timeDiff > 0) {
					return 1;
				} else if (timeDiff < 0) {
					return -1;
				} else if (o1.isStart == o2.isStart) {
					return 0;
				} else if (o1.isStart && !o2.isStart) {
					return 1;
				} else if (!o1.isStart && o2.isStart) {
					return -1;
				}
				return 0;
			}
		});

		// Now it's simple. Any time an protection record starts, one extra computer came online, so
		// add 1 to the total. When a record ends, a computer is no longer being protected, so
		// subtract 1 from the total.
		int peakProtectedComputers = 0;
		int currentPeak = 0;
		for (int i = 0; i < protectionEvents.length; ++i) {
			if (protectionEvents[i].isStart) {
				++currentPeak;
				peakProtectedComputers = Math.max(currentPeak, peakProtectedComputers);
			} else {
				--currentPeak;
			}
			assert currentPeak >= 0;
		}
		
		assert currentPeak == 0;
		
		return peakProtectedComputers;
	}

	/** Simple class to track the beginning or the end of a protection record. */
	static class ProtectionEvent {
		final long time;
		final boolean isStart;
		
		public ProtectionEvent(long time, boolean isStart) {
			this.time = time;
			this.isStart = isStart;
		}

	}

}
