// Copyright (c) Trend Micro Inc. 2014, All Rights Reserved

package com.trendmicro.ds.restapisamples.eventbasedtasks;

import java.util.logging.Level;
import java.util.logging.Logger;

import com.trendmicro.ds.platform.rest.object.Action;
import com.trendmicro.ds.platform.rest.object.Action.Type;
import com.trendmicro.ds.platform.rest.object.Condition;
import com.trendmicro.ds.platform.rest.object.Condition.Field;
import com.trendmicro.ds.platform.rest.object.CreateEventBasedTaskRequest;
import com.trendmicro.ds.platform.rest.object.CreateEventBasedTaskResponse;
import com.trendmicro.ds.platform.rest.object.EventBasedTask;
import com.trendmicro.ds.platform.rest.object.EventBasedTask.EventType;
import com.trendmicro.ds.restapisamples.DeepSecurityClient.ClientException;
import com.trendmicro.ds.restapisamples.DeepSecurityClient;

/**
 * CreateEventBasedTask
 *
 * This sample application shows how to use the Deep Security Manager REST API
 * to create an event-based task that assigns a policy to newly-activated computers
 * based on AWS tags.
 */
public class CreateEventBasedTask {

	private static final String USAGE = "CreateEventBasedTask <DSM URL> <username> <password> [<tenant>] ebtName tagName tagValue policyID\n" +
			"This sample application shows how to use the Deep Security Manager REST API\n" +
			"to create an event-based task that assigns a policy to newly-activated computers\n" +
			"based on AWS tags.";

	private static final Logger logger = Logger.getLogger(CreateEventBasedTask.class.getName());

	/**
	 * The main entry point.
	 *
	 * @param args see USAGE.
	 */
	public static void main(String[] args) {
		if (args.length < 7) {
			logger.warning(USAGE);
			return;
		}

		if (args.length < 8) {
			new CreateEventBasedTask().run(args[0], args[1], args[2], null, args[3], args[4], args[5], args[6]);
		} else {
			new CreateEventBasedTask().run(args[0], args[1], args[2], args[3], args[4], args[5], args[6], args[7]);
		}
	}

	/**
	 * Run the ping sample.
	 *
	 * @param baseUrl base URL for the Deep Security Manager REST API (for example: {@code https://dsm.example.com/rest})
	 * @param username username to use when accessing the Deep Security Manager REST API
	 * @param password password to use when accessing the Deep Security Manager REST API
	 * @param tenant tenant name to use when accessing the Deep Security Manager REST API. If the system is not multi-tenant or if you are accessing the primary tenant, pass {@code null}.
	 */
	private void run(String baseUrl, String username, String password, String tenant, String ebtName, String tagName, String tagValue, String policyIDString) {
		try (DeepSecurityClient client = new DeepSecurityClient(baseUrl, username, password, tenant)) {

			/*
			 * You should <strong>only</strong> use this in a secure development environment, and <strong>never</strong>
			 * use it over an untrusted network. Disabling the trust manager will turn off the validation process that
			 * checks whether the server you are connecting to is the one you expect, and can result in your connection
			 * being hijacked, your credentials stolen, and general catastrophe. Use with great caution!
			 */
			// client.disableTrustManager();

			CreateEventBasedTaskResponse createEventBasedTaskResponse = client.createEventBasedTask(new CreateEventBasedTaskRequest().withTask(new EventBasedTask()
				.withName(ebtName)
				.withType(EventType.AGENT_INITIATED_ACTIVATION)
				.withConditions(new Condition().withField(Field.TAG).withKey(tagName).withValue(tagValue))
				.withActions(new Action().withType(Type.ASSIGN_POLICY).withParameterValue(Integer.valueOf(policyIDString)))
				.withEnabled(true)
			));

			logger.info("Created event-based task " + createEventBasedTaskResponse.getTaskID());

		} catch (DeepSecurityClient.ClientException e) {
			logger.log(Level.SEVERE, "There was an error communicating with the server: " + e.getMessage());
		} catch (Exception e) {
			// Some other error happened, usually related to network communication problems.
			logger.log(Level.SEVERE, "There was an error communicating with the server.", e);
		}
	}

}