// Copyright (c) Trend Micro Inc. 2016, All Rights Reserved

package com.trendmicro.ds.restapisamples.appcontrol;

import java.util.logging.Level;
import java.util.logging.Logger;

import com.trendmicro.ds.appcontrol.rest.object.CreateRulesetRequest;
import com.trendmicro.ds.appcontrol.rest.object.CreateRulesetResponse;
import com.trendmicro.ds.appcontrol.rest.object.DescribeInventoryResponse;
import com.trendmicro.ds.restapisamples.DeepSecurityClient;

/**
 * CreateRuleset
 *
 * This sample application shows how to use the Deep Security Manager REST API
 * to create a ruleset based on a completed inventory scan on a computer.
 */
public class CreateRuleset {

	private static final String USAGE = "CreateRuleset <DSM URL> <username> <password> [<tenant>] inventoryID\n" +
		"This sample application shows how to use the Deep Security Manager REST API\n" +
		"to create a ruleset based on a completed inventory scan on a computer.";

	private static final Logger logger = Logger.getLogger(CreateRuleset.class.getName());

	/**
	 * The main entry point.
	 *
	 * @param args see USAGE.
	 */
	public static void main(String[] args) {
		if (args.length < 4) {
			logger.warning(USAGE);
			return;
		}

		if (args.length == 4) {
			new CreateRuleset().run(args[0], args[1], args[2], null, args[3]);
		} else {
			new CreateRuleset().run(args[0], args[1], args[2], args[3], args[4]);
		}
	}

	private void run(String baseUrl, String username, String password, String tenant, String inventoryIDString) {
		try (DeepSecurityClient client = new DeepSecurityClient(baseUrl, username, password, tenant)) {

			/*
			 * You should <strong>only</strong> use this in a secure development environment, and <strong>never</strong>
			 * use it over an untrusted network. Disabling the trust manager will turn off the validation process that
			 * checks whether the server you are connecting to is the one you expect, and can result in your connection
			 * being hijacked, your credentials stolen, and general catastrophe. Use with great caution!
			 */
			// client.disableTrustManager();

			Long inventoryID = Long.valueOf(inventoryIDString);

			DescribeInventoryResponse describeInventoryResponse = client.describeInventory(inventoryID);

			if (!describeInventoryResponse.getInventoryMetadata().getState().isDone()) {
				logger.warning("Inventory is not yet complete! Try again later.");
				return;
			}

			CreateRulesetResponse createRulesetResponse = client.createRuleset(new CreateRulesetRequest().withInventoryId(inventoryID));

			logger.info("Created ruleset " + createRulesetResponse.getRulesetID() + " from inventory " + inventoryID);

		} catch (DeepSecurityClient.ClientException e) {
			logger.log(Level.SEVERE, "There was an error communicating with the server: " + e.getMessage());
		} catch (Exception e) {
			// Some other error happened, usually related to network communication problems.
			logger.log(Level.SEVERE, "There was an error communicating with the server.", e);
		}
	}

}