// Copyright (c) Trend Micro Inc. 2016, All Rights Reserved

package com.trendmicro.ds.restapisamples;

import java.util.Date;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.trendmicro.ds.platform.rest.object.monitoring.TenantDatabaseUsageElement;
import com.trendmicro.ds.platform.rest.object.monitoring.TenantDatabaseUsageListing;
import com.trendmicro.ds.restapisamples.DeepSecurityClient;

/**
 * Monitoring: This sample application shows a simple use case for accessing the monitoring
 * API. This sample uses an API that is only available to the primary tenant, so does not have
 * a parameter to specify the tenant account.
 */
public class Monitoring {

	private static final String USAGE = "Monitoring <DSM URL> <username> <password>\n" +
		"This sample application can be used to show monitoring information obtained\n" +
		"from Deep Security Manager's REST API.\n";

	private static final Logger logger = Logger.getLogger(Monitoring.class.getName());

	/**
	 * The main entry point.
	 *
	 * @param args see USAGE.
	 */
	public static void main(String[] args) {
		if (args.length < 3) {
			logger.warning(USAGE);
			return;
		}

		new Monitoring().run(args[0], args[1], args[2]);
	}

	/**
	 * Run the sample.
	 *
	 * @param baseUrl base URL for the Deep Security Manager REST API (for example: {@code https://dsm.example.com/rest})
	 * @param username username to use when accessing the Deep Security Manager REST API
	 * @param password password to use when accessing the Deep Security Manager REST API
	 */
	private void run(String baseUrl, String username, String password) {
		try (DeepSecurityClient client = new DeepSecurityClient(baseUrl, username, password, null)) {

			/*
			 * You should <strong>only</strong> use this in a secure development environment, and <strong>never</strong>
			 * use it over an untrusted network. Disabling the trust manager will turn off the validation process that
			 * checks whether the server you are connecting to is the one you expect, and can result in your connection
			 * being hijacked, your credentials stolen, and general catastrophe. Use with great caution!
			 */
			// client.disableTrustManager();

			// Query database size information for the past day to demonstrate
			// some of the functionality available in the monitoring API.
			Date oneDayAgo = new Date(System.currentTimeMillis() - 24 * 60 * 60 * 1000 );

			TenantDatabaseUsageListing usageListing = client.listDatabaseUsage(null, null, oneDayAgo, null);

			for (TenantDatabaseUsageElement usage : usageListing.getUsages()) {
				logger.info(String.format("At %s, tenant %s's database occupied approximately %d bytes\n", usage.getSampleTime(), usage.getTenantName(), Long.valueOf(usage.getDatabaseSize())));
			}

		} catch (DeepSecurityClient.ClientException e) {
			logger.log(Level.SEVERE, "There was an error communicating with the server: " + e.getMessage());
		} catch (Exception e) {
			// Some other error happened, usually related to network communication problems.
			logger.log(Level.SEVERE, "There was an error communicating with the server.", e);
		}
	}

}