// Copyright (c) Trend Micro Inc. 2016, All Rights Reserved

package com.trendmicro.ds.restapisamples.appcontrol;

import java.util.logging.Level;
import java.util.logging.Logger;

import com.trendmicro.ds.appcontrol.rest.object.DescribeInventoryResponse;
import com.trendmicro.ds.appcontrol.rest.object.GetInventoryResponse;
import com.trendmicro.ds.appcontrol.rest.object.InventoryItem;
import com.trendmicro.ds.restapisamples.prerelease.DeepSecurityClient;

/**
 * GetInventory
 *
 * This sample application shows how to use the Deep Security Manager REST API
 * to retrieve the results of a completed inventory scan on a computer.
 */
public class GetInventory {

	private static final String USAGE = "GetInventory <DSM URL> <username> <password> [<tenant>] inventoryID\n" +
		"This sample application shows how to use the Deep Security Manager REST API\n" +
		"to retrieve the results of a completed inventory scan on a computer.";

	private static final Logger logger = Logger.getLogger(GetInventory.class.getName());

	/**
	 * The main entry point.
	 *
	 * @param args see USAGE.
	 */
	public static void main(String[] args) {
		if (args.length < 4) {
			logger.warning(USAGE);
			return;
		}

		if (args.length == 4) {
			new GetInventory().run(args[0], args[1], args[2], null, args[3]);
		} else {
			new GetInventory().run(args[0], args[1], args[2], args[3], args[4]);
		}
	}

	private void run(String baseUrl, String username, String password, String tenant, String inventoryID) {
		try (DeepSecurityClient client = new DeepSecurityClient(baseUrl, username, password, tenant)) {

			/*
			 * You should <strong>only</strong> use this in a secure development environment, and <strong>never</strong>
			 * use it over an untrusted network. Disabling the trust manager will turn off the validation process that
			 * checks whether the server you are connecting to is the one you expect, and can result in your connection
			 * being hijacked, your credentials stolen, and general catastrophe. Use with great caution!
			 */
			// client.disableTrustManager();

			DescribeInventoryResponse describeInventoryResponse = client.describeInventory(Long.valueOf(inventoryID));

			if (!describeInventoryResponse.getInventoryMetadata().getState().isDone()) {
				logger.warning("Inventory is not yet complete! Try again later.");
				return;
			}

			GetInventoryResponse getInventoryResponse = client.getInventory(Long.valueOf(inventoryID));

			for (InventoryItem inventoryItem : getInventoryResponse.getInventoryItems()) {
				// Printing out more of the interesting data in each inventory item is left as an exercise for the reader.
				System.out.println(inventoryItem.getFileName());
			}

		} catch (DeepSecurityClient.ClientException e) {
			logger.log(Level.SEVERE, "There was an error communicating with the server: " + e.getMessage());
		} catch (Exception e) {
			// Some other error happened, usually related to network communication problems.
			logger.log(Level.SEVERE, "There was an error communicating with the server.", e);
		}
	}

}