// Copyright (C) 2016 Trend Micro Inc. All rights reserved.

package com.trendmicro.ds.restapisamples.sslconfigurations;

import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Base64;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.trendmicro.ds.network.rest.object.sslconfiguration.CreateSSLConfigurationRequest;
import com.trendmicro.ds.network.rest.object.sslconfiguration.SSLConfiguration;
import com.trendmicro.ds.network.rest.object.sslconfiguration.SSLCredential;
import com.trendmicro.ds.network.rest.object.sslconfiguration.SSLCredential.Type;
import com.trendmicro.ds.restapisamples.prerelease.DeepSecurityClient;

public class CreateSSLConfiguration {

	private static final String USAGE = "CreateSSLConfiguration <DSM URL> <username> <password> [<tenant>] <hostID> <name> <port> <keyfile.pem>\n" +
			"This sample application shows how to use the Deep Security Manager REST API\n" +
			"to create an SSL configuration on a host.";

	private static final Logger logger = Logger.getLogger(CreateSSLConfiguration.class.getName());

	/**
	 * The main entry point.
	 *
	 * @param args see USAGE.
	 */
	public static void main(String[] args) {
		if (args.length < 6) {
			logger.warning(USAGE);
			return;
		}

		if (args.length < 8) {
			new CreateSSLConfiguration().run(args[0], args[1], args[2], null, args[3], args[4], args[5], args[6]);
		} else {
			new CreateSSLConfiguration().run(args[0], args[1], args[2], args[3], args[4], args[5], args[6], args[7]);
		}
	}

	/**
	 * Run the sample.
	 *
	 * @param baseUrl base URL for the Deep Security Manager REST API (for example: {@code https://dsm.example.com/rest})
	 * @param username username to use when accessing the Deep Security Manager REST API
	 * @param password password to use when accessing the Deep Security Manager REST API
	 * @param tenant tenant name to use when accessing the Deep Security Manager REST API. If the system is not multi-tenant or if you are accessing the primary tenant, pass {@code null}.
	 */
	private void run(String baseUrl, String username, String password, String tenant, String hostID, String name, String port, String keyFile) {
		try (DeepSecurityClient client = new DeepSecurityClient(baseUrl, username, password, tenant)) {

			/*
			 * You should <strong>only</strong> use this in a secure development environment, and <strong>never</strong>
			 * use it over an untrusted network. Disabling the trust manager will turn off the validation process that
			 * checks whether the server you are connecting to is the one you expect, and can result in your connection
			 * being hijacked, your credentials stolen, and general catastrophe. Use with great caution!
			 */
			// client.disableTrustManager();

			client.createSSLConfiguration(Integer.valueOf(hostID), new CreateSSLConfigurationRequest()
				.withSSLConfiguration(new SSLConfiguration()
					.withName(name)
					.withPorts(Integer.valueOf(port))
					.withSSLCredential(new SSLCredential()
						.withName(keyFile)
						.withType(Type.PEM)
						.withPrivateKey(Base64.getEncoder().encodeToString(Files.readAllBytes(Paths.get(keyFile))))
					)
				)
			);

			logger.info("Done.");

		} catch (DeepSecurityClient.ClientException e) {
			logger.log(Level.SEVERE, "There was an error communicating with the server: " + e.getMessage());
		} catch (Exception e) {
			// Some other error happened, usually related to network communication problems.
			logger.log(Level.SEVERE, "There was an error communicating with the server.", e);
		}
	}

}