// Copyright (c) Trend Micro Inc. 2016, All Rights Reserved

package com.trendmicro.ds.restapisamples.appcontrol;

import java.util.logging.Level;
import java.util.logging.Logger;

import com.trendmicro.ds.appcontrol.rest.object.CreateInventoryRequest;
import com.trendmicro.ds.appcontrol.rest.object.CreateInventoryResponse;
import com.trendmicro.ds.appcontrol.rest.object.DescribeInventoryResponse;
import com.trendmicro.ds.restapisamples.prerelease.DeepSecurityClient;

/**
 * BuildInventory
 *
 * This sample application shows how to use the Deep Security Manager REST API
 * to build an inventory on a computer.
 */
public class BuildInventory {

	private static final String USAGE = "BuildInventory <DSM URL> <username> <password> [<tenant>] hostID\n" +
		"This sample application shows how to use the Deep Security Manager REST API\n" +
		"to build an inventory on a computer.";

	private static final Logger logger = Logger.getLogger(BuildInventory.class.getName());

	/**
	 * The main entry point.
	 *
	 * @param args see USAGE.
	 */
	public static void main(String[] args) {
		if (args.length < 4) {
			logger.warning(USAGE);
			return;
		}

		if (args.length == 4) {
			new BuildInventory().run(args[0], args[1], args[2], null, args[3]);
		} else {
			new BuildInventory().run(args[0], args[1], args[2], args[3], args[4]);
		}
	}

	private void run(String baseUrl, String username, String password, String tenant, String hostID) {
		try (DeepSecurityClient client = new DeepSecurityClient(baseUrl, username, password, tenant)) {

			/*
			 * You should <strong>only</strong> use this in a secure development environment, and <strong>never</strong>
			 * use it over an untrusted network. Disabling the trust manager will turn off the validation process that
			 * checks whether the server you are connecting to is the one you expect, and can result in your connection
			 * being hijacked, your credentials stolen, and general catastrophe. Use with great caution!
			 */
			// client.disableTrustManager();

			CreateInventoryResponse createInventoryResponse = client.createInventory(
				new CreateInventoryRequest().withHostID(Integer.valueOf(hostID))
				.withName("SAMPLE APP01-Webserver-v1.0.0")
				.withDescription("SAMPLE Inventory of APP01-WebServer version 1.0.0")
			);

			logger.info("Inventory build requested. Will check on state in 10 seconds...");

			Long inventoryID = createInventoryResponse.getInventoryID();

			DescribeInventoryResponse describeInventoryResponse;
			do {
				Thread.sleep(10000);
				describeInventoryResponse = client.describeInventory(inventoryID);
				logger.info("Building inventory: " + describeInventoryResponse.getInventoryMetadata().getState());
			} while (!describeInventoryResponse.getInventoryMetadata().getState().isDone());

			logger.info("Inventory build complete! Share And Enjoy.");

		} catch (DeepSecurityClient.ClientException e) {
			logger.log(Level.SEVERE, "There was an error communicating with the server: " + e.getMessage());
		} catch (Exception e) {
			// Some other error happened, usually related to network communication problems.
			logger.log(Level.SEVERE, "There was an error communicating with the server.", e);
		}
	}

}