// Copyright (C) 2016 Trend Micro Inc. All rights reserved.

package com.trendmicro.ds.restapisamples.prerelease;

import java.util.List;

import com.trendmicro.ds.platform.rest.api.IManagerInfoAPI;
import com.trendmicro.ds.platform.rest.object.au.ComponentInfoElement;
import com.trendmicro.ds.platform.rest.object.util.FeatureSummaryElement;

/**
 * Sample client helper to simplify access to the Deep Security REST API.
 * <p>
 * This version of the client contains pre-release API calls that are not generally available.
 * If you are part of a private preview program, then you should already have the information
 * you need to use these calls.
 * </p><p>
 * If you are not part of a private preview program, be aware that it's quite likely none of
 * these will work with your Deep Security Manager. If you'd like to be part of our private preview
 * program, get in touch with your local Trend Micro Deep Security sales and support team.
 * </p>
 */
public class DeepSecurityClient extends com.trendmicro.ds.restapisamples.DeepSecurityClient {

	/**
	 * Construct a client to the Deep Security REST API.
	 * <p>
	 * It is a very good idea to use {@code try-with-resources} to manage your client instance and
	 * ensure it is closed properly when you're done with it. For example:
	 * <pre>
	 * try (DeepSecurityClient client = new DeepSecurityClient(url, username, password, account)) {
	 *     client.ping();
	 * }</pre>
	 * will open a client session, check that the Deep Security Manager is active and responding to
	 * requests, and close the client session.
	 * </p>
	 *
	 * @param dsmUrl Base URL for the Deep Security REST API.
	 * @param username User name to use for authentication.
	 * @param password Password to use for authentication.
	 * @param account Tenant account name; required when logging in to a tenant account.
	 */
	public DeepSecurityClient(String dsmUrl, String username, String password, String account) {
		super(dsmUrl, username, password, account);
	}

	/**
	 * Retrieve information about the anti-malware pattern sets currently known to the Deep Security Manager.
	 *
	 * @return information about the anti-malware pattern sets currently known to the Deep Security Manager.
	 */
	public List<ComponentInfoElement> getComponentSummary() throws Exception {
		return withSession((s) -> {
			IManagerInfoAPI managerInfoClient = clientFor(IManagerInfoAPI.class);
			return managerInfoClient.getComponentSummary(s);
		});
	}

	/**
	 * Retrieve a list of feature summary details showing what features are in use in the selected time period.
	 *
	 * @param timescale 1 = the last 7 days; any other value = the last 24 hours.
	 *
	 * @return a list of feature summary details showing what features are in use in the selected time period.
	 */
	public FeatureSummaryElement getFeatureSummary(int timescale) throws Exception {
		return withSession((s) -> {
			IManagerInfoAPI managerInfoClient = clientFor(IManagerInfoAPI.class);
			return managerInfoClient.getFeatureSummary(timescale, s);
		});
	}

}