// Copyright (c) Trend Micro Inc. 2016, All Rights Reserved

package com.trendmicro.ds.restapisamples.prerelease;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.trendmicro.ds.platform.rest.object.au.ComponentInfoElement;
import com.trendmicro.ds.restapisamples.prerelease.DeepSecurityClient;

/**
 * ComponentSummary
 *
 * This sample application shows how to use the Deep Security Manager REST API
 * to retrieve a summary of anti-malware components.
 */
public class ComponentSummary {

	private static final String USAGE = "ComponentSummary <DSM URL> <username> <password> [<tenant>]\n" +
			"This sample application shows how to use the Deep Security Manager REST API\n" +
			"to retrieve anti-malware component summary details.";

	private static final Logger logger = Logger.getLogger(ComponentSummary.class.getName());

	/**
	 * The main entry point.
	 *
	 * @param args see USAGE.
	 */
	public static void main(String[] args) {
		if (args.length < 3) {
			logger.warning(USAGE);
			return;
		}

		new ComponentSummary().run(args[0], args[1], args[2], args.length < 4 ? null : args[3]);
	}

	/**
	 * Run the sample.
	 *
	 * @param baseUrl base URL for the Deep Security Manager REST API (for example: {@code https://dsm.example.com/rest})
	 * @param username username to use when accessing the Deep Security Manager REST API
	 * @param password password to use when accessing the Deep Security Manager REST API
	 * @param tenant tenant name to use when accessing the Deep Security Manager REST API. If the system is not multi-tenant or if you are accessing the primary tenant, pass {@code null}.
	 */
	private void run(String baseUrl, String username, String password, String tenant) {
		try (DeepSecurityClient client = new DeepSecurityClient(baseUrl, username, password, tenant)) {

			/*
			 * You should <strong>only</strong> use this in a secure development environment, and <strong>never</strong>
			 * use it over an untrusted network. Disabling the trust manager will turn off the validation process that
			 * checks whether the server you are connecting to is the one you expect, and can result in your connection
			 * being hijacked, your credentials stolen, and general catastrophe. Use with great caution!
			 */
			// client.disableTrustManager();

			DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

			List<ComponentInfoElement> componentSummary = client.getComponentSummary();

			logger.info("Deep Security Manager component summary");
			for (ComponentInfoElement element : componentSummary) {
				logger.info("Component: " + element.getName() + ", type: " + element.getType() + ", version: " + element.getCurrentVersion() + ", last update: " + dateFormat.format(element.getLastUpdate().getTime()));
			}

		} catch (DeepSecurityClient.ClientException e) {
			logger.log(Level.SEVERE, "There was an error communicating with the server: " + e.getMessage());
		} catch (Exception e) {
			// Some other error happened, usually related to network communication problems.
			logger.log(Level.SEVERE, "There was an error communicating with the server.", e);
		}
	}

}